package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.Assert;

import com.vaadin.data.Item;
import com.vaadin.data.util.PropertysetItem;

import org.easymock.classextension.EasyMock;
import org.junit.Before;
import org.junit.Test;

import org.epo.lifesciences.chepo.viewer.container.ViewerTestUtils;

/**
 * This is a test for {@link AbstractSurrogateIdPageFetchItemStrategy}.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class AbstractSurrogateIdPageFetchItemStrategyTest {

	private AbstractSurrogateIdPageFetchItemStrategy	fetchItemStrategy;

	@Before
	public void setUp() throws SecurityException, NoSuchMethodException {
		fetchItemStrategy = EasyMock
					.createMockBuilder(AbstractSurrogateIdPageFetchItemStrategy.class)
					.addMockedMethod(
								AbstractSurrogateIdPageFetchItemStrategy.class
											.getDeclaredMethod("fetchPage", int.class)).withConstructor()
					.createStrictMock();

		fetchItemStrategy.init(3, 0);
	}

	@Test
	public void testChecks() {
		try {
			fetchItemStrategy.init(0, 1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}

		try {
			fetchItemStrategy.init(1, -1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}
	}

	@Test
	public void testFetchItemStrategy() {
		final List<Item> items = new ArrayList<Item>();

		for (int i = 0; i < 9; i++) {
			items.add(new PropertysetItem());
		}

		EasyMock.expect(fetchItemStrategy.fetchPage(0)).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchPage(1)).andReturn(items.subList(3, 6));
		EasyMock.expect(fetchItemStrategy.fetchPage(0)).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchPage(2)).andReturn(items.subList(6, 9));

		EasyMock.replay(fetchItemStrategy);

		// 1st page:
		fetchItem(items, 0);
		fetchItem(items, 1);
		fetchItem(items, 2);
		fetchItem(items, 0);
		fetchItem(items, 0);
		fetchItem(items, 2);
		fetchItem(items, 1);

		// 2nd page:
		fetchItem(items, 3);
		fetchItem(items, 3);
		fetchItem(items, 5);
		fetchItem(items, 4);
		fetchItem(items, 5);
		fetchItem(items, 4);
		fetchItem(items, 4);
		fetchItem(items, 3);

		// 1st page:
		fetchItem(items, 2);
		fetchItem(items, 0);
		fetchItem(items, 0);
		fetchItem(items, 1);

		// 3rd page:
		fetchItem(items, 6);
		fetchItem(items, 8);
		fetchItem(items, 8);
		fetchItem(items, 7);

		EasyMock.verify(fetchItemStrategy);
	}

	@Test
	public void testFetchItemStrategyWithIndexOutOfBoundsException() {
		final List<Item> items = new ArrayList<Item>();

		for (int i = 0; i < 5; i++) {
			items.add(new PropertysetItem());
		}

		EasyMock.expect(fetchItemStrategy.fetchPage(0)).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchPage(1)).andReturn(items.subList(3, 5));

		EasyMock.replay(fetchItemStrategy);

		// 1st page:
		fetchItem(items, 2);
		fetchItem(items, 0);
		fetchItem(items, 1);

		// 2nd page:
		fetchItem(items, 3);
		fetchItem(items, 3);

		try {
			fetchItemStrategy.getItem(Integer.valueOf(5));
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		fetchItem(items, 4);

		try {
			fetchItemStrategy.getItem(Integer.valueOf(5));
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		EasyMock.verify(fetchItemStrategy);
	}

	private final void fetchItem(List<Item> items, int itemId) {
		Assert.assertTrue(fetchItemStrategy.getItem(Integer.valueOf(itemId)) == items.get(itemId));
	}

	@Test
	public void testToString() {
		Assert.assertNotNull(fetchItemStrategy.toString());
	}

	@Test
	public void testSerialization() throws IOException, ClassNotFoundException {
		final AbstractSurrogateIdPageFetchItemStrategy fetchItemStrategyOriginal = new AbstractSurrogateIdPageFetchItemStrategyInternal();

		fetchItemStrategyOriginal.init(5, 0);

		for (int i = 0; i < 111; i++) {
			fetchItemStrategyOriginal.getItem(Integer.valueOf(i));
		}

		final AbstractSurrogateIdPageFetchItemStrategy fetchItemStrategyDeserialized = ViewerTestUtils
					.serializeDeserializeObject(fetchItemStrategyOriginal);

		// Number of items fetched: 23*5 = 115 < 111
		Assert.assertEquals(
					"AbstractSurrogateIdPageFetchItemStrategyTest.AbstractSurrogateIdPageFetchItemStrategyInternal[pageNumber=22,pageSize=5,pageItems.size=5]",
					fetchItemStrategyOriginal.toString());
		// The cache is cleared:
		Assert.assertEquals(
					"AbstractSurrogateIdPageFetchItemStrategyTest.AbstractSurrogateIdPageFetchItemStrategyInternal[pageNumber=-1,pageSize=5]",
					fetchItemStrategyDeserialized.toString());
	}

	static class AbstractSurrogateIdPageFetchItemStrategyInternal extends AbstractSurrogateIdPageFetchItemStrategy {

		/**
		 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.AbstractSurrogateIdPageFetchItemStrategy#fetchPage(int)
		 */
		@Override
		protected List<Item> fetchPage(int pageNumber) {
			final Item[] items = new Item[getPageSize()];

			for (int i = 0; i < getPageSize(); i++) {
				items[i] = new PropertysetItem();
			}

			return Arrays.asList(items);
		}
	}
}
