package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import java.io.IOException;

import junit.framework.Assert;

import com.vaadin.data.Item;
import com.vaadin.data.util.PropertysetItem;

import org.easymock.classextension.EasyMock;
import org.junit.Before;
import org.junit.Test;

import org.epo.lifesciences.chepo.viewer.container.ViewerTestUtils;

/**
 * This is a test for {@link AbstractLRUFetchItemStrategy}.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class AbstractLRUFetchItemStrategyTest {

	private AbstractLRUFetchItemStrategy<String>	fetchItemStrategy;

	@Before
	public void setUp() throws SecurityException, NoSuchMethodException {
		fetchItemStrategy = EasyMock.createMockBuilder(AbstractLRUFetchItemStrategy.class)
					.addMockedMethod(AbstractLRUFetchItemStrategy.class.getDeclaredMethod("fetchItem", Object.class))
					.withConstructor().createStrictMock();

		fetchItemStrategy.init(1, 0);
	}

	@Test
	public void testChecks() {
		try {
			fetchItemStrategy.init(0, 1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}

		try {
			fetchItemStrategy.init(1, -1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}
	}

	@Test
	public void testFetchItemStrategy() {
		final Item items[] = new Item[]{new PropertysetItem(), new PropertysetItem(), new PropertysetItem(),
				new PropertysetItem()};

		EasyMock.expect(fetchItemStrategy.fetchItem("0")).andReturn(items[0]);
		EasyMock.expect(fetchItemStrategy.fetchItem("1")).andReturn(items[1]);
		EasyMock.expect(fetchItemStrategy.fetchItem("2")).andReturn(items[2]);
		EasyMock.expect(fetchItemStrategy.fetchItem("3")).andReturn(items[3]);
		EasyMock.expect(fetchItemStrategy.fetchItem("1")).andReturn(items[1]);

		EasyMock.replay(fetchItemStrategy);

		fetchItem(items, 0);
		fetchItem(items, 1);
		fetchItem(items, 0);
		fetchItem(items, 1);
		fetchItem(items, 0);
		fetchItem(items, 0);
		fetchItem(items, 1);
		fetchItem(items, 2);
		fetchItem(items, 1);
		fetchItem(items, 1);
		fetchItem(items, 0);
		fetchItem(items, 2);
		fetchItem(items, 2);
		fetchItem(items, 2);
		fetchItem(items, 2);
		fetchItem(items, 0);

		// This kicks out item "1" (most old used) from the cache:
		fetchItem(items, 3);
		fetchItem(items, 3);
		fetchItem(items, 0);
		fetchItem(items, 2);
		fetchItem(items, 2);
		fetchItem(items, 0);
		fetchItem(items, 0);
		fetchItem(items, 3);

		// This will get item "1" back and kick out "2" (most old used):
		fetchItem(items, 1);
		fetchItem(items, 0);
		fetchItem(items, 0);
		fetchItem(items, 3);
		fetchItem(items, 1);
		fetchItem(items, 1);
		fetchItem(items, 3);
		fetchItem(items, 3);
		fetchItem(items, 3);

		EasyMock.verify(fetchItemStrategy);
	}

	private final void fetchItem(Item items[], int itemId) {
		Assert.assertTrue(fetchItemStrategy.getItem(Integer.toString(itemId)) == items[itemId]);
	}

	@Test
	public void testToString() {
		Assert.assertNotNull(fetchItemStrategy.toString());
	}

	@Test
	public void testSerialization() throws IOException, ClassNotFoundException {
		final AbstractLRUFetchItemStrategy<String> fetchItemStrategyOriginal = new AbstractLRUFetchItemStrategyInternal();

		for (int i = 0; i < 111; i++) {
			fetchItemStrategyOriginal.getItem(Integer.toString(i));
		}

		final AbstractLRUFetchItemStrategy<String> fetchItemStrategyDeserialized = ViewerTestUtils
					.serializeDeserializeObject(fetchItemStrategyOriginal);

		Assert.assertEquals(
					"AbstractLRUFetchItemStrategyTest.AbstractLRUFetchItemStrategyInternal[size=100,maxSize=100]",
					fetchItemStrategyOriginal.toString());
		// The cache is cleared:
		Assert.assertEquals(
					"AbstractLRUFetchItemStrategyTest.AbstractLRUFetchItemStrategyInternal[size=0,maxSize=100]",
					fetchItemStrategyDeserialized.toString());
	}

	static class AbstractLRUFetchItemStrategyInternal extends AbstractLRUFetchItemStrategy<String> {
		/**
		 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.AbstractLRUFetchItemStrategy#fetchItem(java.lang.Object)
		 */
		@Override
		protected Item fetchItem(String itemId) {
			return new PropertysetItem();
		}
	}
}
