package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import junit.framework.Assert;

import com.vaadin.data.Container;
import com.vaadin.data.Item;
import com.vaadin.data.util.PropertysetItem;

import org.easymock.IAnswer;
import org.easymock.classextension.EasyMock;
import org.junit.Before;
import org.junit.Test;

import org.epo.lifesciences.chepo.viewer.container.IdReadOnlyContainer;
import org.epo.lifesciences.chepo.viewer.container.ViewerTestUtils;

/**
 * This is a test for {@link AbstractIdPageFetchItemStrategy}.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class AbstractIdPageFetchItemStrategyTest {

	private final List<String>						itemIds	= Arrays.asList(new String[]{"a", "b", "c", "2", "1", "3",
			"+", "-", "*"									});

	private AbstractIdPageFetchItemStrategy<String>	fetchItemStrategy;

	private Container.Indexed						container;

	@Before
	public void setUp() throws SecurityException, NoSuchMethodException {
		fetchItemStrategy = EasyMock.createMockBuilder(AbstractIdPageFetchItemStrategy.class)
					.addMockedMethod(AbstractIdPageFetchItemStrategy.class.getDeclaredMethod("fetchItems", List.class))
					.withConstructor().createStrictMock();

		container = EasyMock.createMock(Container.Indexed.class);

		fetchItemStrategy.init(3, 0);
		fetchItemStrategy.setParentContainer(container);
	}

	@Test
	public void testChecks() {
		try {
			fetchItemStrategy.init(0, 1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}

		try {
			fetchItemStrategy.init(1, -1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}
	}

	@Test
	public void testFetchItemStrategy() {
		final List<Item> items = new ArrayList<Item>();

		for (int i = 0; i < itemIds.size(); i++) {
			items.add(new PropertysetItem());
		}

		EasyMock.expect(Integer.valueOf(container.indexOfId(EasyMock.anyObject()))).andAnswer(new IAnswer<Integer>() {

			/**
			 * @see org.easymock.IAnswer#answer()
			 */
			public Integer answer() throws Throwable {
				return Integer.valueOf(itemIds.indexOf(EasyMock.getCurrentArguments()[0]));
			}
		}).anyTimes();
		EasyMock.expect(container.getItemIds()).andReturn((Collection) itemIds).anyTimes();

		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(0, 3))).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(3, 6))).andReturn(items.subList(3, 6));
		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(0, 3))).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(6, 9))).andReturn(items.subList(6, 9));

		EasyMock.replay(fetchItemStrategy, container);

		// 1st page:
		fetchItem(items, "a");
		fetchItem(items, "b");
		fetchItem(items, "c");
		fetchItem(items, "a");
		fetchItem(items, "a");
		fetchItem(items, "c");
		fetchItem(items, "b");

		// 2nd page:
		fetchItem(items, "2");
		fetchItem(items, "2");
		fetchItem(items, "3");
		fetchItem(items, "1");
		fetchItem(items, "3");
		fetchItem(items, "1");
		fetchItem(items, "1");
		fetchItem(items, "2");

		// 1st page:
		fetchItem(items, "c");
		fetchItem(items, "a");
		fetchItem(items, "a");
		fetchItem(items, "b");

		// 3rd page:
		fetchItem(items, "+");
		fetchItem(items, "*");
		fetchItem(items, "*");
		fetchItem(items, "-");

		EasyMock.verify(fetchItemStrategy, container);
	}

	@Test
	public void testFetchItemStrategyWithIndexOutOfBoundsException() {
		final List<Item> items = new ArrayList<Item>();

		for (int i = 0; i < 5; i++) {
			items.add(new PropertysetItem());
		}

		final List<String> localItemIds = itemIds.subList(0, 5);

		EasyMock.expect(Integer.valueOf(container.indexOfId(EasyMock.anyObject()))).andAnswer(new IAnswer<Integer>() {

			/**
			 * @see org.easymock.IAnswer#answer()
			 */
			public Integer answer() throws Throwable {
				return Integer.valueOf(localItemIds.indexOf(EasyMock.getCurrentArguments()[0]));
			}
		}).anyTimes();
		EasyMock.expect(container.getItemIds()).andReturn((Collection) localItemIds).anyTimes();

		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(0, 3))).andReturn(items.subList(0, 3));
		EasyMock.expect(fetchItemStrategy.fetchItems(itemIds.subList(3, 5))).andReturn(items.subList(3, 5));

		EasyMock.replay(fetchItemStrategy, container);

		// 1st page:
		fetchItem(items, "c");
		fetchItem(items, "a");
		fetchItem(items, "b");

		// 2nd page:
		fetchItem(items, "2");
		fetchItem(items, "2");

		try {
			fetchItemStrategy.getItem("+");
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		fetchItem(items, "1");

		try {
			fetchItemStrategy.getItem("3");
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		EasyMock.verify(fetchItemStrategy, container);
	}

	private final void fetchItem(List<Item> items, String itemId) {
		Assert.assertTrue(fetchItemStrategy.getItem(itemId) == items.get(itemIds.indexOf(itemId)));
	}

	@Test
	public void testToString() {
		Assert.assertNotNull(fetchItemStrategy.toString());
	}

	@Test
	public void testSerialization() throws IOException, ClassNotFoundException {
		final AbstractIdPageFetchItemStrategyInternal fetchItemStrategyOriginal = new AbstractIdPageFetchItemStrategyInternal();

		fetchItemStrategyOriginal.init(5, 0);
		fetchItemStrategyOriginal
					.setParentContainer(new IdReadOnlyContainer<String>(fetchItemStrategyOriginal, itemIds));

		for (int i = 0; i < itemIds.size(); i++) {
			fetchItemStrategyOriginal.getItem(itemIds.get(i));
		}

		final AbstractIdPageFetchItemStrategyInternal fetchItemStrategyDeserialized = ViewerTestUtils
					.serializeDeserializeObject(fetchItemStrategyOriginal);

		Assert.assertEquals(
					"AbstractIdPageFetchItemStrategyTest.AbstractIdPageFetchItemStrategyInternal[pageNumber=1,pageSize=5,pageItems.size=4]",
					fetchItemStrategyOriginal.toString());
		// The cache is cleared:
		Assert.assertEquals(
					"AbstractIdPageFetchItemStrategyTest.AbstractIdPageFetchItemStrategyInternal[pageNumber=-1,pageSize=5]",
					fetchItemStrategyDeserialized.toString());
	}

	static class AbstractIdPageFetchItemStrategyInternal extends AbstractIdPageFetchItemStrategy<String> {

		/**
		 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.AbstractIdPageFetchItemStrategy#fetchItems(java.util.List)
		 */
		@Override
		protected List<Item> fetchItems(List<String> subList) {
			final Item[] items = new Item[subList.size()];

			for (int i = 0; i < subList.size(); i++) {
				items[i] = new PropertysetItem();
			}

			return Arrays.asList(items);
		}
	}
}
