package org.epo.lifesciences.chepo.viewer.container;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

import junit.framework.Assert;

import org.junit.Test;

/**
 * This is a test for {@link RangeList}.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class RangeListTest {

	private static final int	RANGE	= 50;

	@Test
	public void testConstructor() {
		try {
			new RangeList(0);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}

		try {
			new RangeList(0, -1);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}

		try {
			new RangeList(-1, 3);
			Assert.fail("IllegalArgumentException should be thrown");
		}
		catch (IllegalArgumentException e) {
			// should be thrown
		}
	}

	@Test
	public void testOneElementList() {
		testList(new RangeList(1));
	}

	@Test
	public void testMultiElementList() {
		testList(new RangeList(10));
	}

	@Test
	public void testRangeList() {
		testList(new RangeList(12, 24));
	}

	private static void testList(RangeList actual) {
		testReadOnlyList(actual);
		testReadOnlyIterators(actual);
		testSubListChecks(actual);

		final List<Integer> expected = new ArrayList<Integer>();

		for (int i = actual.getFromIndex(); i <= actual.getToIndex(); i++) {
			expected.add(Integer.valueOf(i));
		}

		compareLists(expected, actual);
	}

	private static void testReadOnlyList(List<Integer> list) {
		final Collection<Integer> value = new ArrayList<Integer>();

		for (int i = -RANGE; i <= RANGE; i++) {
			try {
				list.set(i + RANGE, Integer.valueOf(1));
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			try {
				list.add(i, Integer.valueOf(1));
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			try {
				list.remove(Integer.valueOf(i));
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			try {
				list.remove(i);
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			value.add(Integer.valueOf(i));

			try {
				list.addAll(i, value);
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}
		}

		try {
			list.add(Integer.valueOf(1));
			Assert.fail("UnsupportedOperationException should be thrown");
		}
		catch (UnsupportedOperationException e) {
			// should be thrown
		}

		try {
			list.addAll(value);
			Assert.fail("UnsupportedOperationException should be thrown");
		}
		catch (UnsupportedOperationException e) {
			// should be thrown
		}

		try {
			list.removeAll(value);
			Assert.fail("UnsupportedOperationException should be thrown");
		}
		catch (UnsupportedOperationException e) {
			// should be thrown
		}

		try {
			list.retainAll(value);
			Assert.fail("UnsupportedOperationException should be thrown");
		}
		catch (UnsupportedOperationException e) {
			// should be thrown
		}

		try {
			list.clear();
			Assert.fail("UnsupportedOperationException should be thrown");
		}
		catch (UnsupportedOperationException e) {
			// should be thrown
		}
	}

	private static void testReadOnlyIterators(List<Integer> list) {
		final Iterator<Integer> iter = list.iterator();

		while (iter.hasNext()) {
			try {
				iter.remove();
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			iter.next();
		}

		final ListIterator<Integer> listIter = list.listIterator();

		while (listIter.hasNext()) {
			try {
				listIter.remove();
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			try {
				listIter.add(Integer.valueOf(1));
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			try {
				listIter.set(Integer.valueOf(1));
				Assert.fail("UnsupportedOperationException should be thrown");
			}
			catch (UnsupportedOperationException e) {
				// should be thrown
			}

			listIter.next();
		}
	}

	private static void testSubListChecks(RangeList actual) {
		try {
			actual.subList(0, 0);
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		try {
			actual.subList(-1, 1);
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		try {
			actual.subList(0, -1);
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		try {
			actual.subList(2, 1);
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		try {
			actual.subList(actual.size() - 1, actual.size());
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}

		try {
			actual.subList(actual.size(), actual.size() + 1);
			Assert.fail("IndexOutOfBoundsException should be thrown");
		}
		catch (IndexOutOfBoundsException e) {
			// should be thrown
		}
	}

	private static void compareLists(final List<Integer> expected, RangeList actual) {
		compareGeneral(expected, actual);

		// General iterator test:
		final ListIterator<Integer> expectedListIterator = expected.listIterator();
		final ListIterator<Integer> actualListIterator = actual.listIterator();

		compareForwardIteration(expectedListIterator, actualListIterator);
		compareBackwardIteration(expectedListIterator, actualListIterator);

		// Forward iterator test:
		compareForwardIteration(expected.listIterator((actual.getToIndex() - actual.getFromIndex()) / 2),
					actual.listIterator((actual.getToIndex() - actual.getFromIndex()) / 2));

		// Backward iterator test:
		compareBackwardIteration(expected.listIterator((actual.getToIndex() - actual.getFromIndex()) / 2),
					actual.listIterator((actual.getToIndex() - actual.getFromIndex()) / 2));

		compareToArray(expected, actual);

		compareSubLists(expected, actual);
	}

	private static void compareGeneral(List<Integer> expected, List<Integer> actual) {
		Assert.assertEquals(expected.size(), actual.size());
		Assert.assertEquals(expected.isEmpty(), actual.isEmpty());

		Assert.assertEquals(-1, actual.indexOf(""));

		for (int i = -RANGE; i <= RANGE; i++) {
			Assert.assertEquals(expected.indexOf(Integer.valueOf(i)), actual.indexOf(Integer.valueOf(i)));
			Assert.assertEquals(expected.lastIndexOf(Integer.valueOf(i)), actual.lastIndexOf(Integer.valueOf(i)));
			Assert.assertEquals(expected.contains(Integer.valueOf(i)), actual.contains(Integer.valueOf(i)));

			Integer expectedValue = null;
			Integer actualValue = null;

			try {
				expectedValue = expected.get(i);
			}
			catch (IndexOutOfBoundsException e) {}

			try {
				actualValue = actual.get(i);
			}
			catch (IndexOutOfBoundsException e) {}

			Assert.assertEquals("For index " + i, expectedValue, actualValue);

			ListIterator<Integer> expectedListIterator = null;
			ListIterator<Integer> actualListIterator = null;

			try {
				expectedListIterator = expected.listIterator(i);
			}
			catch (IndexOutOfBoundsException e) {}

			try {
				actualListIterator = actual.listIterator(i);
			}
			catch (IndexOutOfBoundsException e) {}

			Assert.assertEquals("For index " + i, expectedListIterator == null, actualListIterator == null);
		}

		final Collection<Integer> list = new ArrayList<Integer>();

		for (int i = 0; i <= RANGE; i++) {
			list.add(Integer.valueOf(i));

			Assert.assertEquals(expected.containsAll(list), actual.containsAll(list));
		}
	}

	private static void compareForwardIteration(ListIterator<Integer> expectedListIterator,
				ListIterator<Integer> actualListIterator) {
		Assert.assertEquals(expectedListIterator.hasNext(), actualListIterator.hasNext());
		Assert.assertEquals(expectedListIterator.nextIndex(), actualListIterator.nextIndex());
		Assert.assertEquals(expectedListIterator.hasPrevious(), actualListIterator.hasPrevious());
		Assert.assertEquals(expectedListIterator.previousIndex(), actualListIterator.previousIndex());

		while (expectedListIterator.hasNext()) {
			Assert.assertEquals(expectedListIterator.next(), actualListIterator.next());
			Assert.assertEquals(expectedListIterator.hasNext(), actualListIterator.hasNext());
			Assert.assertEquals(expectedListIterator.nextIndex(), actualListIterator.nextIndex());
			Assert.assertEquals(expectedListIterator.hasPrevious(), actualListIterator.hasPrevious());
			Assert.assertEquals(expectedListIterator.previousIndex(), actualListIterator.previousIndex());
		}

		try {
			actualListIterator.next();
			Assert.fail("NoSuchElementException should be thrown");
		}
		catch (NoSuchElementException e) {
			// should be thrown
		}
	}

	private static void compareBackwardIteration(ListIterator<Integer> expectedListIterator,
				ListIterator<Integer> actualListIterator) {
		Assert.assertEquals(expectedListIterator.hasNext(), actualListIterator.hasNext());
		Assert.assertEquals(expectedListIterator.nextIndex(), actualListIterator.nextIndex());
		Assert.assertEquals(expectedListIterator.hasPrevious(), actualListIterator.hasPrevious());
		Assert.assertEquals(expectedListIterator.previousIndex(), actualListIterator.previousIndex());

		while (expectedListIterator.hasPrevious()) {
			Assert.assertEquals(expectedListIterator.previous(), actualListIterator.previous());
			Assert.assertEquals(expectedListIterator.hasNext(), actualListIterator.hasNext());
			Assert.assertEquals(expectedListIterator.nextIndex(), actualListIterator.nextIndex());
			Assert.assertEquals(expectedListIterator.hasPrevious(), actualListIterator.hasPrevious());
			Assert.assertEquals(expectedListIterator.previousIndex(), actualListIterator.previousIndex());
		}

		try {
			actualListIterator.previous();
			Assert.fail("NoSuchElementException should be thrown");
		}
		catch (NoSuchElementException e) {
			// should be thrown
		}
	}

	private static void compareToArray(final List<Integer> expected, List<Integer> actual) {
		Arrays.equals(expected.toArray(), actual.toArray());

		try {
			expected.toArray(null);
			Assert.fail("NullPointerException should be thrown");
		}
		catch (NullPointerException e) {
			// should be thrown
		}

		try {
			actual.toArray(null);
			Assert.fail("NullPointerException should be thrown");
		}
		catch (NullPointerException e) {
			// should be thrown
		}

		for (int i = 0; i <= RANGE; i++) {
			{
				Number expectedArray[] = new Number[i];
				Number actualArray[] = new Number[i];

				Arrays.fill(expectedArray, Integer.valueOf(RANGE));
				Arrays.fill(actualArray, Integer.valueOf(RANGE));

				Assert.assertTrue(Arrays.equals(expected.toArray(expectedArray), actual.toArray(actualArray)));
				Assert.assertTrue(Arrays.equals(expectedArray, actualArray));
			}

			{
				Integer expectedArray[] = new Integer[i];
				Integer actualArray[] = new Integer[i];

				Arrays.fill(expectedArray, Integer.valueOf(RANGE));
				Arrays.fill(actualArray, Integer.valueOf(RANGE));

				Assert.assertTrue(Arrays.equals(expected.toArray(expectedArray), actual.toArray(actualArray)));
				Assert.assertTrue(Arrays.equals(expectedArray, actualArray));
			}

			{
				String expectedArray[] = new String[i];
				String actualArray[] = new String[i];

				try {
					expected.toArray(expectedArray);
					Assert.fail("ArrayStoreException should be thrown");
				}
				catch (ArrayStoreException e) {
					// should be thrown
				}

				try {
					actual.toArray(actualArray);
					Assert.fail("ArrayStoreException should be thrown");
				}
				catch (ArrayStoreException e) {
					// should be thrown
				}

				Arrays.equals(expectedArray, actualArray);
			}
		}
	}

	private static void compareSubLists(List<Integer> expected, final RangeList actual) {
		if (actual.size() > 2) {
			int middle = actual.size() / 2;

			compareLists(expected.subList(0, middle - 1), actual.subList(0, middle - 1));
			compareLists(expected.subList(middle, expected.size() - 1), actual.subList(middle, expected.size() - 1));
		}
	}
}
