package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import java.util.List;

import com.vaadin.data.Item;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

import org.epo.lifesciences.chepo.viewer.container.SurrogateIdReadOnlyContainer;

/**
 * This implementation of {@link FetchItemStrategy} fetches items page by page.<br>
 * Note: it is suitable only for {@link SurrogateIdReadOnlyContainer} as it relies on the fact that item ID is the same
 * as row index.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public abstract class AbstractSurrogateIdPageFetchItemStrategy implements FetchItemStrategy<Integer> {

	private static final int		DEFAULT_PAGE_SIZE	= 100;

	private int						pageSize			= DEFAULT_PAGE_SIZE;

	private int						currentPageNumber	= -1;

	private transient List<Item>	currentPageItems;

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#init(int, double)
	 */
	public void init(int pageSize, double cacheRate) {
		if (pageSize <= 0 || cacheRate < 0) {
			throw new IllegalArgumentException("page size and cache rate should be positive");
		}

		this.pageSize = pageSize;
	}

	public int getPageSize() {
		return pageSize;
	}

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#getItem(java.lang.Object)
	 */
	public Item getItem(Integer itemId) {
		final int rowNumber = itemId.intValue();
		final int pageNumber = rowNumber / pageSize;

		if (pageNumber != currentPageNumber) {
			currentPageItems = fetchPage(pageNumber);
			currentPageNumber = pageNumber;
		}

		return currentPageItems.get(rowNumber - (currentPageNumber * pageSize));
	}

	/**
	 * Fetches all items for the given page (page numbers start from 0).
	 */
	protected abstract List<Item> fetchPage(int pageNumber);

	/**
	 * Recover transient cache after deserialization.
	 */
	Object readResolve() {
		currentPageNumber = -1;

		return this;
	}

	@Override
	public String toString() {
		final ToStringBuilder sb = new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE);

		sb.append("pageNumber", currentPageNumber);
		sb.append("pageSize", pageSize);

		if (currentPageItems != null) {
			sb.append("pageItems.size", currentPageItems.size());
		}

		return sb.toString();
	}
}
