package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import com.vaadin.data.Item;

import org.apache.commons.collections.map.LRUMap;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

/**
 * Provides the functionality for the container, that holds only the subset of most frequently used data, and lazily
 * loads the rows on demand.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public abstract class AbstractLRUFetchItemStrategy<IdType> implements FetchItemStrategy<IdType> {

	private static final int	DEFAULT_SIZE	= 100;

	private transient LRUMap	map				= new LRUMap(DEFAULT_SIZE);

	private int					maxSize			= DEFAULT_SIZE;

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#init(int, float)
	 */
	public void init(int pageSize, double cacheRate) {
		if (pageSize <= 0 || cacheRate < 0) {
			throw new IllegalArgumentException("page size and cache rate should be positive");
		}

		// cache minimum 3 visual pages:
		this.maxSize = (int) (pageSize * (3 + cacheRate));

		if (map.maxSize() != maxSize) {
			map = new LRUMap(maxSize);
		}
	}

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#getItem(java.lang.Object)
	 */
	public Item getItem(IdType itemId) {
		Item item = (Item) map.get(itemId);

		if (item == null) {
			item = fetchItem(itemId);
			map.put(itemId, item);
		}

		return item;
	}

	protected abstract Item fetchItem(IdType itemId);

	/**
	 * Recover transient cache after deserialization.
	 */
	Object readResolve() {
		map = new LRUMap(maxSize);

		return this;
	}

	@Override
	public String toString() {
		final ToStringBuilder sb = new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE);

		sb.append("size", map.size());
		sb.append("maxSize", map.maxSize());

		return sb.toString();
	}
}
