package org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy;

import java.util.List;

import com.vaadin.data.Container;
import com.vaadin.data.Item;

import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.ToStringStyle;

/**
 * This implementation of {@link FetchItemStrategy} fetches items page by page.<br>
 * This strategy can be serialized, of referred container can be serialized.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public abstract class AbstractIdPageFetchItemStrategy<IdType> implements FetchItemStrategy<IdType> {

	private static final int		DEFAULT_PAGE_SIZE	= 100;

	private int						pageSize			= DEFAULT_PAGE_SIZE;

	private Container.Indexed		parentContainer;

	private int						currentPageNumber	= -1;

	private transient List<Item>	currentPageItems;

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#init(int, double)
	 */
	public void init(int pageSize, double cacheRate) {
		if (pageSize <= 0 || cacheRate < 0) {
			throw new IllegalArgumentException("page size and cache rate should be positive");
		}

		this.pageSize = pageSize;
	}

	public void setParentContainer(Container.Indexed parentContainer) {
		this.parentContainer = parentContainer;
	}

	/**
	 * @see org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy#getItem(java.lang.Object)
	 */
	public Item getItem(IdType itemId) {
		final int rowNumber = parentContainer.indexOfId(itemId);

		if (rowNumber == -1) {
			throw new IndexOutOfBoundsException("The given item ID " + itemId + " is unknown");
		}

		final int pageNumber = rowNumber / pageSize;
		final int pageOffset = rowNumber % pageSize;

		if (pageNumber != currentPageNumber) {
			final List<IdType> itemIds = (List<IdType>) parentContainer.getItemIds();
			final int startIndex = rowNumber - pageOffset;
			final int endIndex = Math.min(itemIds.size(), startIndex + pageSize);

			currentPageItems = fetchItems(itemIds.subList(startIndex, endIndex));

			if (currentPageItems.size() != endIndex - startIndex) {
				throw new IllegalStateException("The fetched page size is " + currentPageItems.size()
							+ " but should be " + (endIndex - startIndex));
			}

			currentPageNumber = pageNumber;
		}

		return currentPageItems.get(pageOffset);
	}

	/**
	 * Fetches all items for the given range of item IDs. The size of the resulting list should be he same as the size
	 * of given IDs list.
	 */
	protected abstract List<Item> fetchItems(List<IdType> itemIds);

	/**
	 * Recover transient cache after deserialization.
	 */
	Object readResolve() {
		currentPageNumber = -1;

		return this;
	}

	@Override
	public String toString() {
		final ToStringBuilder sb = new ToStringBuilder(this, ToStringStyle.SHORT_PREFIX_STYLE);

		sb.append("pageNumber", currentPageNumber);
		sb.append("pageSize", pageSize);

		if (currentPageItems != null) {
			sb.append("pageItems.size", currentPageItems.size());
		}

		return sb.toString();
	}
}
