package org.epo.lifesciences.chepo.viewer.container;

import java.util.Collection;

import com.vaadin.data.Container;

import org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy;

/**
 * Provides the implementation of {@link Container} interface, which is based on the set of ordered in some way items.
 * The item ID in this container matches the index of the row (and hence is surrogate ID).<br>
 * This container implementation is good in case when you have millions of items, for which you cannot fetch even the
 * list of item IDs as it will not fit into memory.<br>
 * The provided {@link FetchItemStrategy} implementation should be able to fetch the item by it's index.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class SurrogateIdReadOnlyContainer extends AbstractReadOnlyContainer<Integer> {

	private int	size;

	public SurrogateIdReadOnlyContainer(FetchItemStrategy<Integer> fetchItemStrategy, int size) {
		super(fetchItemStrategy);

		this.size = size;
	}

	/**
	 * @see com.vaadin.data.Container#containsId(java.lang.Object)
	 */
	public boolean containsId(Object itemId) {
		final int index = ((Integer) itemId).intValue();

		return index >= 0 && index < size;
	}

	/**
	 * @see com.vaadin.data.Container#size()
	 */
	public int size() {
		return size;
	}

	/**
	 * @see com.vaadin.data.Container#getItemIds()
	 */
	public Collection<?> getItemIds() {
		return new RangeList(size);
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#indexOfId(java.lang.Object)
	 */
	public int indexOfId(Object itemId) {
		return ((Integer) itemId).intValue();
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#getIdByIndex(int)
	 */
	public Object getIdByIndex(int index) {
		return Integer.valueOf(index);
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#firstItemId()
	 */
	public Object firstItemId() {
		return Integer.valueOf(0);
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#lastItemId()
	 */
	public Object lastItemId() {
		return Integer.valueOf(size - 1);
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#isFirstId(java.lang.Object)
	 */
	public boolean isFirstId(Object o) {
		return ((Integer) o).intValue() == 0;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#isLastId(java.lang.Object)
	 */
	public boolean isLastId(Object o) {
		return ((Integer) o).intValue() == size - 1;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#nextItemId(java.lang.Object)
	 */
	public Object nextItemId(Object o) {
		final int index = ((Integer) o).intValue();

		if (index < size - 1) {
			return Integer.valueOf(index + 1);
		}

		return null;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#prevItemId(java.lang.Object)
	 */
	public Object prevItemId(Object o) {
		final int rowNumber = ((Integer) o).intValue();

		if (rowNumber > 0) {
			return Integer.valueOf(rowNumber - 1);
		}

		return null;
	}
}
