package org.epo.lifesciences.chepo.viewer.container;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;

/**
 * This is a implementation of {@link List} interface, that contains the range of integers.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class RangeList implements List<Integer> {

	private final int	fromIndex;

	private final int	toIndex;

	/**
	 * Creates a list of integers from 0 to size-1.
	 */
	public RangeList(int size) {
		if (size <= 0) {
			throw new IllegalArgumentException("size should be positive");
		}

		this.fromIndex = 0;
		this.toIndex = size - 1;
	}

	/**
	 * Creates a list of integers with a given range, inclusive.
	 */
	public RangeList(int from, int to) {
		if (from < 0) {
			throw new IllegalArgumentException("fromIndex should be non-negative");
		}

		if (to < from) {
			throw new IllegalArgumentException("toIndex should be not less then fromIndex");
		}

		this.fromIndex = from;
		this.toIndex = to;
	}

	protected int getFromIndex() {
		return fromIndex;
	}

	protected int getToIndex() {
		return toIndex;
	}

	/**
	 * @see java.util.List#size()
	 */
	public int size() {
		return toIndex - fromIndex + 1;
	}

	/**
	 * @see java.util.List#isEmpty()
	 */
	public boolean isEmpty() {
		return size() == 0;
	}

	/**
	 * @see java.util.List#get(int)
	 */
	public Integer get(int index) {
		if (index >= size() || index < 0) {
			throw new IndexOutOfBoundsException(index + " (index) should be non-negative and less than " + size()
						+ " (size)");
		}

		return Integer.valueOf(index + fromIndex);
	}

	/**
	 * @see java.util.List#set(int, java.lang.Object)
	 */
	public Integer set(int index, Integer element) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#indexOf(java.lang.Object)
	 */
	public int indexOf(Object o) {
		if (o instanceof Number) {
			final int index = ((Number) o).intValue();

			if (index >= fromIndex && index <= toIndex) {
				return index - fromIndex;
			}
		}

		return -1;
	}

	/**
	 * @see java.util.List#lastIndexOf(java.lang.Object)
	 */
	public int lastIndexOf(Object o) {
		// Hence all elements are unique:
		return indexOf(o);
	}

	/**
	 * @see java.util.List#add(java.lang.Object)
	 */
	public boolean add(Integer e) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#add(int, java.lang.Object)
	 */
	public void add(int index, Integer element) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#addAll(java.util.Collection)
	 */
	public boolean addAll(Collection<? extends Integer> c) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#addAll(int, java.util.Collection)
	 */
	public boolean addAll(int index, Collection<? extends Integer> c) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#contains(java.lang.Object)
	 */
	public boolean contains(Object o) {
		return indexOf(o) != -1;
	}

	/**
	 * @see java.util.List#containsAll(java.util.Collection)
	 */
	public boolean containsAll(Collection<?> c) {
		for (final Object o : c) {
			if (!contains(o)) {
				return false;
			}
		}

		return true;
	}

	/**
	 * @see java.util.List#remove(java.lang.Object)
	 */
	public boolean remove(Object o) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#remove(int)
	 */
	public Integer remove(int index) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#removeAll(java.util.Collection)
	 */
	public boolean removeAll(Collection<?> c) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#retainAll(java.util.Collection)
	 */
	public boolean retainAll(Collection<?> c) {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#clear()
	 */
	public void clear() {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see java.util.List#iterator()
	 */
	public Iterator<Integer> iterator() {
		return listIterator();
	}

	/**
	 * @see java.util.List#listIterator()
	 */
	public ListIterator<Integer> listIterator() {
		return new FixedSizeListIterator();
	}

	/**
	 * @see java.util.List#listIterator(int)
	 */
	public ListIterator<Integer> listIterator(int index) {
		if (index > size() || index < 0) {
			throw new IndexOutOfBoundsException(index + " (index) should be non-negative and less than " + size()
						+ " (size)");
		}

		return new FixedSizeListIterator(index);
	}

	/**
	 * @see java.util.List#subList(int, int)
	 */
	public RangeList subList(int fromIndex, int toIndex) {
		if (fromIndex < 0) {
			throw new IndexOutOfBoundsException("fromIndex should be non-negative");
		}

		if (toIndex <= fromIndex) {
			throw new IndexOutOfBoundsException("toIndex should be greater then fromIndex");
		}

		if (toIndex >= size()) {
			throw new IndexOutOfBoundsException("toIndex should be less then list size");
		}

		return new RangeList(this.fromIndex + fromIndex, this.fromIndex + toIndex - 1);
	}

	private class FixedSizeListIterator implements ListIterator<Integer> {

		private int	index;

		protected FixedSizeListIterator() {
			this(0);
		}

		protected FixedSizeListIterator(int index) {
			this.index = index;
		}

		/**
		 * @see java.util.ListIterator#hasNext()
		 */
		public boolean hasNext() {
			return index < size();
		}

		/**
		 * @see java.util.ListIterator#next()
		 */
		public Integer next() {
			if (index >= size()) {
				throw new NoSuchElementException();
			}

			return get(index++);
		}

		/**
		 * @see java.util.ListIterator#hasPrevious()
		 */
		public boolean hasPrevious() {
			return index > 0;
		}

		/**
		 * @see java.util.ListIterator#previous()
		 */
		public Integer previous() {
			if (index <= 0) {
				throw new NoSuchElementException();
			}

			return get(--index);
		}

		/**
		 * @see java.util.ListIterator#nextIndex()
		 */
		public int nextIndex() {
			return index;
		}

		/**
		 * @see java.util.ListIterator#previousIndex()
		 */
		public int previousIndex() {
			return index - 1;
		}

		/**
		 * @see java.util.ListIterator#remove()
		 */
		public void remove() {
			throw new UnsupportedOperationException();
		}

		/**
		 * @see java.util.ListIterator#set(java.lang.Object)
		 */
		public void set(Integer e) {
			throw new UnsupportedOperationException();
		}

		/**
		 * @see java.util.ListIterator#add(java.lang.Object)
		 */
		public void add(Integer e) {
			throw new UnsupportedOperationException();
		}
	}

	/**
	 * @see java.util.List#toArray()
	 */
	public Object[] toArray() {
		final int size = size();
		final Integer array[] = new Integer[size];

		for (int i = 0; i < size; i++) {
			array[i] = Integer.valueOf(i + fromIndex);
		}

		return array;
	}

	/**
	 * @see java.util.List#toArray(T[])
	 */
	public <T> T[] toArray(T[] a) {
		if (!a.getClass().isAssignableFrom(new Integer[0].getClass())) {
			throw new ArrayStoreException("Can't cast " + a.getClass() + " to Integer[]");
		}

		final int size = size();

		if (a.length < size) {
			return (T[]) toArray();
		}

		final Number[] result = (new Number[0].getClass().cast(a));

		for (int i = 0; i < size; i++) {
			result[i] = Integer.valueOf(i + fromIndex);
		}

		// Put a null-marker if possible:
		if (size < a.length) {
			a[size] = null;
		}

		return a;
	}
}
