package org.epo.lifesciences.chepo.viewer.container;

import java.util.List;

import com.vaadin.data.Container;

import org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy;

/**
 * Provides the implementation of {@link Container} interface, which is based on the set of ordered in some way items.
 * Item IDs for this container are provided in a constructor.<br>
 * This container implementation is good in case when you have not so many items, for which can fetch the item IDs into
 * memory.<br>
 * The provided {@link FetchItemStrategy} implementation should be able to fetch the item by it's real ID.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public class IdReadOnlyContainer<IdType> extends AbstractReadOnlyContainer<IdType> {

	private final List<IdType>	itemIds;

	public IdReadOnlyContainer(FetchItemStrategy<IdType> fetchItemStrategy, List<IdType> itemIds) {
		super(fetchItemStrategy);

		this.itemIds = itemIds;
	}

	/**
	 * @see com.vaadin.data.Container#getItemIds()
	 */
	public List<IdType> getItemIds() {
		return itemIds;
	}

	/**
	 * @see com.vaadin.data.Container#size()
	 */
	public int size() {
		return itemIds.size();
	}

	/**
	 * @see com.vaadin.data.Container#containsId(java.lang.Object)
	 */
	public boolean containsId(Object itemId) {
		return itemIds.contains(itemId);
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#indexOfId(java.lang.Object)
	 */
	public int indexOfId(Object itemId) {
		return itemIds.indexOf(itemId);
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#getIdByIndex(int)
	 */
	public IdType getIdByIndex(int index) {
		return itemIds.get(index);
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#nextItemId(java.lang.Object)
	 */
	public IdType nextItemId(Object itemId) {
		int index = itemIds.indexOf(itemId);

		if (index >= 0 && index < itemIds.size() - 1) {
			return itemIds.get(index + 1);
		}

		return null;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#prevItemId(java.lang.Object)
	 */
	public IdType prevItemId(Object itemId) {
		int index = itemIds.indexOf(itemId);

		if (index > 0) {
			return itemIds.get(index - 1);
		}

		return null;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#firstItemId()
	 */
	public IdType firstItemId() {
		if (!itemIds.isEmpty()) {
			return itemIds.get(0);
		}

		return null;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#lastItemId()
	 */
	public IdType lastItemId() {
		if (!itemIds.isEmpty()) {
			return itemIds.get(itemIds.size() - 1);
		}

		return null;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#isFirstId(java.lang.Object)
	 */
	public boolean isFirstId(Object itemId) {
		return itemIds.indexOf(itemId) == 0;
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#isLastId(java.lang.Object)
	 */
	public boolean isLastId(Object itemId) {
		return itemIds.indexOf(itemId) == itemIds.size() - 1;
	}
}
