package org.epo.lifesciences.chepo.viewer.container;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.data.Container;
import com.vaadin.data.Item;
import com.vaadin.data.Property;

import org.epo.lifesciences.chepo.viewer.container.fetchitemstrategy.FetchItemStrategy;

/**
 * The basic class implementing {@link Container} interface, that includes the functionality to work with properties and
 * change listeners.
 * 
 * @author <a href="mailto:dkatsubo@epo.org">Dmitry Katsubo</a>
 */
public abstract class AbstractReadOnlyContainer<IdType> implements Serializable, Container, Container.Indexed {

	private final FetchItemStrategy<IdType>				fetchItemStrategy;

	/**
	 * Map propertyID-to-PropertyDescription
	 */
	protected final Map<Object, PropertyDescription>	properties			= new HashMap<Object, PropertyDescription>();

	private static final long							serialVersionUID	= 1L;

	public AbstractReadOnlyContainer(FetchItemStrategy<IdType> fetchItemStrategy) {
		this.fetchItemStrategy = fetchItemStrategy;
	}

	/**
	 * @see com.vaadin.data.Container#getItem(java.lang.Object)
	 */
	public Item getItem(Object itemId) {
		return fetchItemStrategy.getItem((IdType) itemId);
	}

	/**
	 * This class contains subset of property fields.
	 */
	private static class PropertyDescription {
		public Class<?>	type;
		public Object	defaultValue;

		protected PropertyDescription(Class<?> type, Object defaultValue) {
			this.type = type;
			this.defaultValue = defaultValue;
		}
	}

	/**
	 * This class is introduced to defer the call to {@link Container#getItem(Object)} to the most later time (when
	 * property value is requested, not the property itself).<br>
	 * Note: in most cases properties are either {@link com.vaadin.data.util.MethodProperty} or
	 * {@link com.vaadin.data.util.ObjectProperty}, which support all interfaces listed here.
	 */
	private class ReadOnlyContainerProperty
				implements
					Property,
					Property.ValueChangeNotifier,
					Property.ReadOnlyStatusChangeNotifier {

		private final Object	itemId;
		private final Object	propertyId;

		protected ReadOnlyContainerProperty(Object itemId, Object propertyId) {
			this.itemId = itemId;
			this.propertyId = propertyId;
		}

		private Property getPropertyInternal() {
			return getItem(itemId).getItemProperty(propertyId);
		}

		/**
		 * @see com.vaadin.data.Property#getValue()
		 */
		public Object getValue() {
			return getPropertyInternal().getValue();
		}

		/**
		 * @see com.vaadin.data.Property#setValue(java.lang.Object)
		 */
		public void setValue(Object newValue) throws ReadOnlyException, ConversionException {
			getPropertyInternal().setValue(newValue);
		}

		/**
		 * @see com.vaadin.data.Property#getType()
		 */
		public Class<?> getType() {
			return properties.get(propertyId).type;
		}

		/**
		 * @see com.vaadin.data.Property#isReadOnly()
		 */
		public boolean isReadOnly() {
			return getPropertyInternal().isReadOnly();
		}

		/**
		 * @see com.vaadin.data.Property#setReadOnly(boolean)
		 */
		public void setReadOnly(boolean newStatus) {
			getPropertyInternal().setReadOnly(newStatus);
		}

		/**
		 * @see com.vaadin.data.Property.ReadOnlyStatusChangeNotifier#addListener(com.vaadin.data.Property.ReadOnlyStatusChangeListener)
		 */
		public void addListener(ReadOnlyStatusChangeListener listener) {
			((Property.ReadOnlyStatusChangeNotifier) getPropertyInternal()).addListener(listener);
		}

		/**
		 * @see com.vaadin.data.Property.ReadOnlyStatusChangeNotifier#removeListener(com.vaadin.data.Property.ReadOnlyStatusChangeListener)
		 */
		public void removeListener(ReadOnlyStatusChangeListener listener) {
			((Property.ReadOnlyStatusChangeNotifier) getPropertyInternal()).removeListener(listener);
		}

		/**
		 * @see com.vaadin.data.Property.ValueChangeNotifier#addListener(com.vaadin.data.Property.ValueChangeListener)
		 */
		public void addListener(ValueChangeListener listener) {
			((Property.ValueChangeNotifier) getPropertyInternal()).addListener(listener);
		}

		/**
		 * @see com.vaadin.data.Property.ValueChangeNotifier#removeListener(com.vaadin.data.Property.ValueChangeListener)
		 */
		public void removeListener(ValueChangeListener listener) {
			((Property.ValueChangeNotifier) getPropertyInternal()).removeListener(listener);
		}

		/**
		 * @see java.lang.Object#toString()
		 */
		@Override
		public String toString() {
			return getPropertyInternal().toString();
		}
	}

	/**
	 * @see com.vaadin.data.Container#addItem()
	 */
	public Object addItem() throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container#removeItem(java.lang.Object)
	 */
	public boolean removeItem(Object o) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container#removeAllItems()
	 */
	public boolean removeAllItems() throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container#addItem(java.lang.Object)
	 */
	public Item addItem(Object o) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#addItemAt(int)
	 */
	public Object addItemAt(int i) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container.Indexed#addItemAt(int, java.lang.Object)
	 */
	public Item addItemAt(int i, Object o) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#addItemAfter(java.lang.Object)
	 */
	public Object addItemAfter(Object o) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container.Ordered#addItemAfter(java.lang.Object, java.lang.Object)
	 */
	public Item addItemAfter(Object o, Object o1) throws UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	/**
	 * @see com.vaadin.data.Container#addContainerProperty(java.lang.Object, java.lang.Class, java.lang.Object)
	 */
	public boolean addContainerProperty(Object propertyId, Class<?> type, Object defaultValue)
				throws UnsupportedOperationException {
		if (propertyId == null || type == null) {
			return false;
		}

		if (properties.containsKey(propertyId)) {
			return false;
		}

		properties.put(propertyId, new PropertyDescription(type, defaultValue));

		return true;
	}

	public void addContainerProperty(Object propertyId, Class<?> type) {
		addContainerProperty(propertyId, type, null);
	}

	/**
	 * @see com.vaadin.data.Container#removeContainerProperty(java.lang.Object)
	 */
	public boolean removeContainerProperty(Object propertyId) throws UnsupportedOperationException {
		if (!properties.containsKey(propertyId)) {
			return false;
		}

		properties.remove(propertyId);

		return true;
	}

	/**
	 * @see com.vaadin.data.Container#getContainerProperty(java.lang.Object, java.lang.Object)
	 */
	public Property getContainerProperty(Object itemId, Object propertyId) {
		if (itemId == null) {
			return null;
		}

		return getItem(itemId).getItemProperty(propertyId);
	}

	/**
	 * @see com.vaadin.data.Container#getContainerPropertyIds()
	 */
	public Collection<?> getContainerPropertyIds() {
		return Collections.unmodifiableCollection(properties.keySet());
	}

	/**
	 * @see com.vaadin.data.Container#getType(java.lang.Object)
	 */
	public Class<?> getType(Object propertyId) {
		final PropertyDescription propertyDescription = properties.get(propertyId);

		if (propertyDescription != null) {
			return propertyDescription.type;
		}

		return null;
	}
}
