/*
 * Copyright 2008 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.util.tools;

import com.google.gwt.dev.shell.BrowserChannel;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;

/**
 * Command-line tool to generate a C header file for OOPHM extensions.
 * 
 * Output is generated to the file specified as the first argument or stdout if
 * no arguments are supplied.
 */
public class OophmHeaderFileGenerator {

  /**
   * Generates C/C++ header file for select Java constants.
   * 
   * Usage: [outputFile]
   * 
   * @param args
   */
  public static void main(String[] args) throws IOException {
    PrintStream str = System.out;
    if (args.length == 1) {
      str = new PrintStream(new File(args[0]));
    }
    str.println("/*");
    str.println(" * DO NOT EDIT");
    str.println(" * Generated by "
        + OophmHeaderFileGenerator.class.getCanonicalName());
    str.println(" */");
    str.println();
    str.println("/* from BrowserChannel.BROWSERCHANNEL_PROTOCOL_VERSION */");
    str.println("#define BROWSERCHANNEL_PROTOCOL_VERSION "
        + BrowserChannel.BROWSERCHANNEL_PROTOCOL_VERSION);
    dumpEnums(str, BrowserChannel.SessionHandler.SpecialDispatchId.class);
    dumpEnums(str, BrowserChannel.MessageType.class);
    dumpEnums(str, BrowserChannel.Value.ValueType.class);
  }

  /**
   * Append a C-style constant version of the supplied Java name to a
   * StringBuffer.
   * 
   * Converts camel-case identifiers to all-caps with underscores.
   * 
   * <pre>
   *   ie: MyMessageType => MY_MESSAGE_TYPE
   *       MY_CONSTANT => MY_CONSTANT
   *       MyHTTPRequest => MY_HTTPREQUEST
   * </pre>
   * 
   * @param buf StringBuffer instance to append to
   * @param javaName Java-style name
   */
  private static void appendCName(StringBuffer buf, String javaName) {
    boolean needUnderscore = false;
    for (char ch : javaName.toCharArray()) {
      if (Character.isUpperCase(ch)) {
        if (needUnderscore) {
          buf.append('_');
        }
        needUnderscore = false;
      } else if (ch == '_') {
        needUnderscore = false;
      } else {
        needUnderscore = true;
      }
      buf.append(Character.toUpperCase(ch));
    }
  }

  /**
   * Write a #define line for a single enum constant.
   * 
   * @param str PrintStream to write to
   * @param enumName name of the Enum containing this value
   * @param constName name of the individual enum constant to write
   * @param ordinal the ordinal value of this enum constant
   */
  private static void dumpEnum(PrintStream str, String enumName,
      String constName, int ordinal) {
    StringBuffer buf = new StringBuffer();
    buf.append("#define ");
    appendCName(buf, enumName);
    buf.append('_');
    appendCName(buf, constName);
    buf.append(' ');
    buf.append(ordinal);
    str.println(buf.toString());
  }

  /**
   * Write #defines all enum constants for a given enum class.
   * 
   * @param <E> enum type parameter
   * @param str stream to write to
   * @param e enum class object
   */
  private static <E extends Enum<E>> void dumpEnums(PrintStream str, Class<E> e) {
    str.println();
    str.println("/* from " + e.getCanonicalName() + " */");
    E[] constants = e.getEnumConstants();
    String enumName = e.getSimpleName();
    for (E constant : constants) {
      dumpEnum(str, enumName, constant.name(), constant.ordinal());
    }
  }
}
